var VisualProfiler;
(function (VisualProfiler) {
    var GlobalRuler = (function () {
        function GlobalRuler(totalRange) {
            this._totalRange = totalRange;
            this._activeRange = this._selection = this._totalRange;
            this._publisher = new Common.Publisher([
                GlobalRuler.SelectionChangedEventType, 
                GlobalRuler.ActiveRangeChangedEventType
            ]);
        }
        GlobalRuler.SelectionChangedEventType = "selectionChanged";
        GlobalRuler.ActiveRangeChangedEventType = "activeRangeChanged";
        Object.defineProperty(GlobalRuler.prototype, "totalRange", {
            get: function () {
                return this._totalRange;
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(GlobalRuler.prototype, "selection", {
            get: function () {
                return this._selection;
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(GlobalRuler.prototype, "activeRange", {
            get: function () {
                return this._activeRange;
            },
            enumerable: true,
            configurable: true
        });
        GlobalRuler.prototype.setSelection = function (newSelection, isIntermittent) {
            if (typeof isIntermittent === "undefined") { isIntermittent = false; }
            if(!this._selection.equals(newSelection) || !isIntermittent) {
                var begin = VisualProfiler.TimeStamp.fromNanoseconds(Math.max(newSelection.begin.nsec, this._activeRange.begin.nsec));
                var end = VisualProfiler.TimeStamp.fromNanoseconds(Math.min(newSelection.end.nsec, this._activeRange.end.nsec));
                this._selection = new VisualProfiler.TimeSpan(begin, end);
                this._publisher.invokeListener({
                    type: GlobalRuler.SelectionChangedEventType,
                    data: {
                        isIntermittent: isIntermittent,
                        newSelection: newSelection
                    }
                });
                Plugin.VS.Internal.CodeMarkers.fire(VisualProfiler.CodeMarkerValues.perfBrowserTools_VisualProfilerUserSelectedTimeSlice);
            }
        };
        GlobalRuler.prototype.setActiveRange = function (newRange) {
            if(!this._activeRange.equals(newRange)) {
                this._activeRange = newRange;
                this._publisher.invokeListener({
                    type: GlobalRuler.ActiveRangeChangedEventType
                });
            }
        };
        GlobalRuler.prototype.addEventListener = function (eventType, func) {
            this._publisher.addEventListener(eventType, func);
        };
        GlobalRuler.prototype.removeEventListener = function (eventType, func) {
            this._publisher.removeEventListener(eventType, func);
        };
        return GlobalRuler;
    })();
    VisualProfiler.GlobalRuler = GlobalRuler;    
    var MarkEventModel = (function () {
        function MarkEventModel(session) {
            this._session = session;
        }
        MarkEventModel.prototype.getMarkEvents = function (timeRange, category) {
            return this._session.queryMarkEvents(timeRange.begin.nsec, timeRange.end.nsec, category);
        };
        MarkEventModel.prototype.getMarkTooltip = function (mark) {
            var eventData = this._session.getEventData(mark.threadId, mark.time);
            var tooltip = mark.name;
            if(typeof eventData.EventName !== "undefined") {
                var eventPayload = eventData.EventName;
                if((eventPayload.length > 20) && (eventPayload.substring(0, 20) === "VisualProfiler:mark,")) {
                    tooltip = eventPayload.slice(20);
                }
            }
            tooltip += Plugin.Resources.getString("RulerMarkTooltipLabel", VisualProfiler.FormattingHelpers.getPrettyPrintTime(VisualProfiler.TimeStamp.fromNanoseconds(mark.time)));
            return tooltip;
        };
        return MarkEventModel;
    })();
    VisualProfiler.MarkEventModel = MarkEventModel;    
    var RulerViewModel = (function () {
        function RulerViewModel(globalRuler, markEventModel) {
            this.leftPadding = 40;
            this.rightPadding = 40;
            this._globalRuler = globalRuler;
            this._markEventModel = markEventModel;
        }
        RulerViewModel.prototype.getMarks = function (category) {
            return this._markEventModel.getMarkEvents(this._globalRuler.totalRange, category);
        };
        RulerViewModel.prototype.getMarkTooltip = function (mark) {
            return this._markEventModel.getMarkTooltip(mark);
        };
        Object.defineProperty(RulerViewModel.prototype, "globalRuler", {
            get: function () {
                return this._globalRuler;
            },
            enumerable: true,
            configurable: true
        });
        RulerViewModel.prototype.setSelectionRange = function (newSelection, isIntermittent) {
            if (typeof isIntermittent === "undefined") { isIntermittent = false; }
            this._globalRuler.setSelection(newSelection, isIntermittent);
        };
        RulerViewModel.prototype.setActiveRange = function (newRange) {
            this._globalRuler.setActiveRange(newRange);
        };
        return RulerViewModel;
    })();
    VisualProfiler.RulerViewModel = RulerViewModel;    
    var RulerView = (function () {
        function RulerView(containerId) {
            this._parentContainer = document.getElementById(containerId);
            if(!this._parentContainer) {
                throw new Error(Plugin.Resources.getErrorString("JSPerf.1034"));
            }
        }
        Object.defineProperty(RulerView.prototype, "minSelectionWidthInPixels", {
            set: function (value) {
                this._minSelectionWidthInPixels = value;
            },
            enumerable: true,
            configurable: true
        });
        Object.defineProperty(RulerView.prototype, "viewModel", {
            set: function (model) {
                this.unregisterViewModelEvents();
                this._viewModel = model;
                this.registerViewModelEvents();
            },
            enumerable: true,
            configurable: true
        });
        RulerView.prototype.render = function () {
            var _this = this;
            this._parentContainer.innerHTML = "";
            if(this._rulerControl) {
                this._rulerControl.deinitialize();
            }
            var rulerConfig = new VisualProfiler.Graph.RulerConfig(this._parentContainer.id);
            rulerConfig.doubleSlider.timeRange = this._viewModel.globalRuler.totalRange;
            rulerConfig.doubleSlider.bar.left = this._viewModel.leftPadding;
            rulerConfig.doubleSlider.bar.right = this._viewModel.rightPadding;
            if(this._minSelectionWidthInPixels) {
                rulerConfig.doubleSlider.minimumRangeInPixel = this._minSelectionWidthInPixels;
            }
            var lifecycleData = [];
            var userMarkData = [];
            var lifecycleMarks = this._viewModel.getMarks(0);
            for(var markIndex = 0; markIndex < lifecycleMarks.length; markIndex++) {
                (function (mark) {
                    var markData = new VisualProfiler.Graph.MarkData(VisualProfiler.TimeStamp.fromNanoseconds(mark.time), function () {
                        return _this._viewModel.getMarkTooltip(mark);
                    });
                    lifecycleData.push(markData);
                })(lifecycleMarks[markIndex]);
            }
            var userMarks = this._viewModel.getMarks(1);
            for(var markIndex = 0; markIndex < userMarks.length; markIndex++) {
                (function (mark) {
                    var markData = new VisualProfiler.Graph.MarkData(VisualProfiler.TimeStamp.fromNanoseconds(mark.time), function () {
                        return _this._viewModel.getMarkTooltip(mark);
                    });
                    userMarkData.push(markData);
                })(userMarks[markIndex]);
            }
            rulerConfig.doubleSlider.markSeries = [
                {
                    id: "LifecycleMark",
                    label: Plugin.Resources.getString("RulerLifecycleMarkLabel"),
                    data: lifecycleData,
                    type: VisualProfiler.Graph.MarkType.LifeCycleEvent
                }, 
                {
                    id: "UserMark",
                    label: Plugin.Resources.getString("RulerUserMarkLabel"),
                    data: userMarkData,
                    type: VisualProfiler.Graph.MarkType.UserMark
                }
            ];
            this._rulerControl = new VisualProfiler.Graph.Ruler(rulerConfig);
            this._rulerControl.render();
            this._rulerControl.addEventListener(VisualProfiler.Graph.RulerEvents.Filter, this.onRulerFilterChange.bind(this));
        };
        RulerView.prototype.unregisterViewModelEvents = function () {
            if(this._viewModel) {
                this._viewModel.globalRuler.removeEventListener(GlobalRuler.SelectionChangedEventType, this.onGlobalRulerSelectionChanged.bind(this));
                this._viewModel.globalRuler.removeEventListener(GlobalRuler.ActiveRangeChangedEventType, this.onGlobalRulerActiveRangeChange.bind(this));
            }
        };
        RulerView.prototype.registerViewModelEvents = function () {
            if(this._viewModel) {
                this._viewModel.globalRuler.addEventListener(GlobalRuler.SelectionChangedEventType, this.onGlobalRulerSelectionChanged.bind(this));
                this._viewModel.globalRuler.addEventListener(GlobalRuler.ActiveRangeChangedEventType, this.onGlobalRulerActiveRangeChange.bind(this));
            }
        };
        RulerView.prototype.onRulerFilterChange = function (args) {
            this.adjustSelection(args.data.position, args.data.isIntermittent);
        };
        RulerView.prototype.adjustSelection = function (selection, isIntermittent) {
            if (typeof isIntermittent === "undefined") { isIntermittent = false; }
            if(this._viewModel) {
                this._viewModel.setSelectionRange(selection, isIntermittent);
            }
        };
        RulerView.prototype.onGlobalRulerActiveRangeChange = function (args) {
            if(this._viewModel && this._rulerControl) {
                this._rulerControl.setTimeRange(this._viewModel.globalRuler.activeRange);
                this._rulerControl.clearSelection();
            }
        };
        RulerView.prototype.onGlobalRulerSelectionChanged = function (args) {
            if(this._viewModel && this._rulerControl) {
                this._rulerControl.setHandlePosition(this._viewModel.globalRuler.selection);
            }
        };
        return RulerView;
    })();
    VisualProfiler.RulerView = RulerView;    
})(VisualProfiler || (VisualProfiler = {}));
//@ sourceMappingURL=RulerView.js.map

// SIG // Begin signature block
// SIG // MIIajwYJKoZIhvcNAQcCoIIagDCCGnwCAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFGHrkW/SH9tt
// SIG // 3J6dQzzENAM9FTNXoIIVeTCCBLowggOioAMCAQICCmEC
// SIG // jkIAAAAAAB8wDQYJKoZIhvcNAQEFBQAwdzELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWljcm9zb2Z0IFRp
// SIG // bWUtU3RhbXAgUENBMB4XDTEyMDEwOTIyMjU1OFoXDTEz
// SIG // MDQwOTIyMjU1OFowgbMxCzAJBgNVBAYTAlVTMRMwEQYD
// SIG // VQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25k
// SIG // MR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24x
// SIG // DTALBgNVBAsTBE1PUFIxJzAlBgNVBAsTHm5DaXBoZXIg
// SIG // RFNFIEVTTjpGNTI4LTM3NzctOEE3NjElMCMGA1UEAxMc
// SIG // TWljcm9zb2Z0IFRpbWUtU3RhbXAgU2VydmljZTCCASIw
// SIG // DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAJbsjkdN
// SIG // VMJclYDXTgs9v5dDw0vjYGcRLwFNDNjRRi8QQN4LpFBS
// SIG // EogLQ3otP+5IbmbHkeYDym7sealqI5vNYp7NaqQ/56ND
// SIG // /2JHobS6RPrfQMGFVH7ooKcsQyObUh8yNfT+mlafjWN3
// SIG // ezCeCjOFchvKSsjMJc3bXREux7CM8Y9DSEcFtXogC+Xz
// SIG // 78G69LPYzTiP+yGqPQpthRfQyueGA8Azg7UlxMxanMTD
// SIG // 2mIlTVMlFGGP+xvg7PdHxoBF5jVTIzZ3yrDdmCs5wHU1
// SIG // D92BTCE9djDFsrBlcylIJ9jC0rCER7t4utV0A97XSxn3
// SIG // U9542ob3YYgmM7RHxqBUiBUrLHUCAwEAAaOCAQkwggEF
// SIG // MB0GA1UdDgQWBBQv6EbIaNNuT7Ig0N6JTvFH7kjB8jAf
// SIG // BgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7syuwwzWzDzBU
// SIG // BgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vY3JsLm1pY3Jv
// SIG // c29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsGAQUFBwEBBEww
// SIG // SjBIBggrBgEFBQcwAoY8aHR0cDovL3d3dy5taWNyb3Nv
// SIG // ZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRUaW1lU3Rh
// SIG // bXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsGAQUFBwMIMA0G
// SIG // CSqGSIb3DQEBBQUAA4IBAQBz/30unc2NiCt8feNeFXHp
// SIG // aGLwCLZDVsRcSi1o2PlIEZHzEZyF7BLUVKB1qTihWX91
// SIG // 7sb1NNhUpOLQzHyXq5N1MJcHHQRTLDZ/f/FAHgybgOIS
// SIG // CiA6McAHdWfg+jSc7Ij7VxzlWGIgkEUvXUWpyI6zfHJt
// SIG // ECfFS9hvoqgSs201I2f6LNslLbldsR4F50MoPpwFdnfx
// SIG // Jd4FRxlt3kmFodpKSwhGITWodTZMt7MIqt+3K9m+Kmr9
// SIG // 3zUXzD8Mx90Gz06UJGMgCy4krl9DRBJ6XN0326RFs5E6
// SIG // Eld940fGZtPPnEZW9EwHseAMqtX21Tyi4LXU+Bx+BFUQ
// SIG // axj0kc1Rp5VlMIIE7DCCA9SgAwIBAgITMwAAALARrwqL
// SIG // 0Duf3QABAAAAsDANBgkqhkiG9w0BAQUFADB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQTAeFw0xMzAxMjQyMjMzMzla
// SIG // Fw0xNDA0MjQyMjMzMzlaMIGDMQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYDVQQDExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEB
// SIG // AQUAA4IBDwAwggEKAoIBAQDor1yiIA34KHy8BXt/re7r
// SIG // dqwoUz8620B9s44z5lc/pVEVNFSlz7SLqT+oN+EtUO01
// SIG // Fk7vTXrbE3aIsCzwWVyp6+HXKXXkG4Unm/P4LZ5BNisL
// SIG // QPu+O7q5XHWTFlJLyjPFN7Dz636o9UEVXAhlHSE38Cy6
// SIG // IgsQsRCddyKFhHxPuRuQsPWj/ov0DJpOoPXJCiHiquMB
// SIG // Nkf9L4JqgQP1qTXclFed+0vUDoLbOI8S/uPWenSIZOFi
// SIG // xCUuKq6dGB8OHrbCryS0DlC83hyTXEmmebW22875cHso
// SIG // AYS4KinPv6kFBeHgD3FN/a1cI4Mp68fFSsjoJ4TTfsZD
// SIG // C5UABbFPZXHFAgMBAAGjggFgMIIBXDATBgNVHSUEDDAK
// SIG // BggrBgEFBQcDAzAdBgNVHQ4EFgQUWXGmWjNN2pgHgP+E
// SIG // Hr6H+XIyQfIwUQYDVR0RBEowSKRGMEQxDTALBgNVBAsT
// SIG // BE1PUFIxMzAxBgNVBAUTKjMxNTk1KzRmYWYwYjcxLWFk
// SIG // MzctNGFhMy1hNjcxLTc2YmMwNTIzNDRhZDAfBgNVHSME
// SIG // GDAWgBTLEejK0rQWWAHJNy4zFha5TJoKHzBWBgNVHR8E
// SIG // TzBNMEugSaBHhkVodHRwOi8vY3JsLm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NybC9wcm9kdWN0cy9NaWNDb2RTaWdQQ0Ff
// SIG // MDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUHAQEETjBMMEoG
// SIG // CCsGAQUFBzAChj5odHRwOi8vd3d3Lm1pY3Jvc29mdC5j
// SIG // b20vcGtpL2NlcnRzL01pY0NvZFNpZ1BDQV8wOC0zMS0y
// SIG // MDEwLmNydDANBgkqhkiG9w0BAQUFAAOCAQEAMdduKhJX
// SIG // M4HVncbr+TrURE0Inu5e32pbt3nPApy8dmiekKGcC8N/
// SIG // oozxTbqVOfsN4OGb9F0kDxuNiBU6fNutzrPJbLo5LEV9
// SIG // JBFUJjANDf9H6gMH5eRmXSx7nR2pEPocsHTyT2lrnqkk
// SIG // hNrtlqDfc6TvahqsS2Ke8XzAFH9IzU2yRPnwPJNtQtjo
// SIG // fOYXoJtoaAko+QKX7xEDumdSrcHps3Om0mPNSuI+5PNO
// SIG // /f+h4LsCEztdIN5VP6OukEAxOHUoXgSpRm3m9Xp5QL0f
// SIG // zehF1a7iXT71dcfmZmNgzNWahIeNJDD37zTQYx2xQmdK
// SIG // Dku/Og7vtpU6pzjkJZIIpohmgjCCBbwwggOkoAMCAQIC
// SIG // CmEzJhoAAAAAADEwDQYJKoZIhvcNAQEFBQAwXzETMBEG
// SIG // CgmSJomT8ixkARkWA2NvbTEZMBcGCgmSJomT8ixkARkW
// SIG // CW1pY3Jvc29mdDEtMCsGA1UEAxMkTWljcm9zb2Z0IFJv
// SIG // b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5MB4XDTEwMDgz
// SIG // MTIyMTkzMloXDTIwMDgzMTIyMjkzMloweTELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWljcm9zb2Z0IENv
// SIG // ZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqGSIb3DQEBAQUA
// SIG // A4IBDwAwggEKAoIBAQCycllcGTBkvx2aYCAgQpl2U2w+
// SIG // G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/3sJCTiPVcgDb
// SIG // NVcKicquIEn08GisTUuNpb15S3GbRwfa/SXfnXWIz6pz
// SIG // RH/XgdvzvfI2pMlcRdyvrT3gKGiXGqelcnNW8ReU5P01
// SIG // lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJpL9oZC/6SdCni
// SIG // di9U3RQwWfjSjWL9y8lfRjFQuScT5EAwz3IpECgixzdO
// SIG // PaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn0i1i8UU956wI
// SIG // APZGoZ7RW4wmU+h6qkryRs83PDietHdcpReejcsRj1Y8
// SIG // wawJXwPTAgMBAAGjggFeMIIBWjAPBgNVHRMBAf8EBTAD
// SIG // AQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGCNxUBBAUCAwEA
// SIG // ATAjBgkrBgEEAYI3FQIEFgQU/dExTtMmipXhmGA7qDFv
// SIG // pjy82C0wGQYJKwYBBAGCNxQCBAweCgBTAHUAYgBDAEEw
// SIG // HwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8KuEKU5VZ5KQw
// SIG // UAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC5taWNy
// SIG // b3NvZnQuY29tL3BraS9jcmwvcHJvZHVjdHMvbWljcm9z
// SIG // b2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUFBwEBBEgwRjBE
// SIG // BggrBgEFBQcwAoY4aHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNyb3NvZnRSb290Q2VydC5j
// SIG // cnQwDQYJKoZIhvcNAQEFBQADggIBAFk5Pn8mRq/rb0Cx
// SIG // MrVq6w4vbqhJ9+tfde1MOy3XQ60L/svpLTGjI8x8UJiA
// SIG // IV2sPS9MuqKoVpzjcLu4tPh5tUly9z7qQX/K4QwXacul
// SIG // nCAt+gtQxFbNLeNK0rxw56gNogOlVuC4iktX8pVCnPHz
// SIG // 7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y4k74jKHK6BOl
// SIG // kU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wdzaKMvSeyeWNW
// SIG // RKJRzfnpo1hW3ZsCRUQvX/TartSCMm78pJUT5Otp56mi
// SIG // LL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q70eFW6NB4lhhc
// SIG // yTUWX92THUmOLb6tNEQc7hAVGgBd3TVbIc6YxwnuhQ6M
// SIG // T20OE049fClInHLR82zKwexwo1eSV32UjaAbSANa98+j
// SIG // Zwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKNMxZlHg6K3RDe
// SIG // ZPRvzkbU0xfpecQEtNP7LN8fip6sCvsTJ0Ct5PnhqX9G
// SIG // uwdgR2VgQE6wQuxO7bN2edgKNAltHIAxH+IOVN3lofvl
// SIG // RxCtZJj/UBYufL8FIXrilUEnacOTj5XJjdibIa4NXJzw
// SIG // oq6GaIMMai27dmsAHZat8hZ79haDJLmIz2qoRzEvmtzj
// SIG // cT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq/2mbluIQqBC0
// SIG // N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZoNAAAAAAAHDAN
// SIG // BgkqhkiG9w0BAQUFADBfMRMwEQYKCZImiZPyLGQBGRYD
// SIG // Y29tMRkwFwYKCZImiZPyLGQBGRYJbWljcm9zb2Z0MS0w
// SIG // KwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBDZXJ0aWZpY2F0
// SIG // ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1MzA5WhcNMjEw
// SIG // NDAzMTMwMzA5WjB3MQswCQYDVQQGEwJVUzETMBEGA1UE
// SIG // CBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEe
// SIG // MBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0aW9uMSEw
// SIG // HwYDVQQDExhNaWNyb3NvZnQgVGltZS1TdGFtcCBQQ0Ew
// SIG // ggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCf
// SIG // oWyx39tIkip8ay4Z4b3i48WZUSNQrc7dGE4kD+7Rp9FM
// SIG // rXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr6Hu97IkHD/cO
// SIG // BJjwicwfyzMkh53y9GccLPx754gd6udOo6HBI1PKjfpF
// SIG // zwnQXq/QsEIEovmmbJNn1yjcRlOwhtDlKEYuJ6yGT1VS
// SIG // DOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd++NIT8wi3U21S
// SIG // tEWQn0gASkdmEScpZqiX5NMGgUqi+YSnEUcUCYKfhO1V
// SIG // eP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68eeEExd8yb3zuD
// SIG // k6FhArUdDbH895uyAc4iS1T/+QXDwiALAgMBAAGjggGr
// SIG // MIIBpzAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBQj
// SIG // NPjZUkZwCu1A+3b7syuwwzWzDzALBgNVHQ8EBAMCAYYw
// SIG // EAYJKwYBBAGCNxUBBAMCAQAwgZgGA1UdIwSBkDCBjYAU
// SIG // DqyCYEBWJ5flJRP8KuEKU5VZ5KShY6RhMF8xEzARBgoJ
// SIG // kiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJk/IsZAEZFglt
// SIG // aWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jvc29mdCBSb290
// SIG // IENlcnRpZmljYXRlIEF1dGhvcml0eYIQea0WoUqgpa1M
// SIG // c1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9taWNyb3NvZnRyb290Y2VydC5jcmwwVAYIKwYBBQUH
// SIG // AQEESDBGMEQGCCsGAQUFBzAChjhodHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY3Jvc29mdFJv
// SIG // b3RDZXJ0LmNydDATBgNVHSUEDDAKBggrBgEFBQcDCDAN
// SIG // BgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wDRDbd6bStd9vO
// SIG // eVFNAbEudHFbbQwTq86+e4+4LtQSooxtYrhXAstOIBNQ
// SIG // md16QOJXu69YmhzhHQGGrLt48ovQ7DsB7uK+jwoFyI1I
// SIG // 4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mRKiQicPv2/OR4
// SIG // mS4N9wficLwYTp2OawpylbihOZxnLcVRDupiXD8WmIsg
// SIG // P+IHGjL5zDFKdjE9K3ILyOpwPf+FChPfwgphjvDXuBfr
// SIG // Tot/xTUrXqO/67x9C0J71FNyIe4wyrt4ZVxbARcKFA7S
// SIG // 2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGysOUzU9nm/qhh6
// SIG // YinvopspNAZ3GmLJPR5tH4LwC8csu89Ds+X57H2146So
// SIG // dDW4TsVxIxImdgs8UoxxWkZDFLyzs7BNZ8ifQv+AeSGA
// SIG // nhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU2DKATCYqSCRf
// SIG // WupW76bemZ3KOm+9gSd0BhHudiG/m4LBJ1S2sWo9iaF2
// SIG // YbRuoROmv6pH8BJv/YoybLL+31HIjCPJZr2dHYcSZAI9
// SIG // La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCleKuzoJZ1GtmS
// SIG // hxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/Jmu5J4PcBZW+J
// SIG // C33Iacjmbuqnl84xKf8OxVtc2E0bodj6L54/LlUWa8kT
// SIG // o/0xggSCMIIEfgIBATCBkDB5MQswCQYDVQQGEwJVUzET
// SIG // MBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVk
// SIG // bW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBvcmF0
// SIG // aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQgQ29kZSBTaWdu
// SIG // aW5nIFBDQQITMwAAALARrwqL0Duf3QABAAAAsDAJBgUr
// SIG // DgMCGgUAoIGkMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3
// SIG // AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEV
// SIG // MCMGCSqGSIb3DQEJBDEWBBTWuYdyE3kgc//5kqFq90hY
// SIG // TM7BqjBEBgorBgEEAYI3AgEMMTYwNKAagBgAUgB1AGwA
// SIG // ZQByAFYAaQBlAHcALgBqAHOhFoAUaHR0cDovL21pY3Jv
// SIG // c29mdC5jb20wDQYJKoZIhvcNAQEBBQAEggEAf7+1Yda+
// SIG // TEQcAkDQd2EOGHxh/f4ZKQH7SdVVsBQ1tfIDjo61yMxk
// SIG // GR2pWPt+PEYaZKHsZfJa6O6QSyFtWUjD6jCCChEIbB5E
// SIG // paWwGH7HpChCVFHt7kxlw6cfHmgoIDNcvh+h+qzzPxuf
// SIG // 651RQbwZc/HEtYjc2VwU4ikLVjUlEEtfNvQ6FW4Fd1v1
// SIG // YZz1qZZOR+pa+sMmOGp7IsVQwYCgsAm9W5DvfVA4iuae
// SIG // uBusiQAH1VW+4x8JZJqesTfNZgvCJtfrxj6PJLbsrDTP
// SIG // BVQvtJVbw+NnLVe44cAIbACVhMV8XXuyitGBCfHhA2Sk
// SIG // gYVptKt0okeDJ8O6LDviMBzwTqGCAh8wggIbBgkqhkiG
// SIG // 9w0BCQYxggIMMIICCAIBATCBhTB3MQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
// SIG // UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
// SIG // cmF0aW9uMSEwHwYDVQQDExhNaWNyb3NvZnQgVGltZS1T
// SIG // dGFtcCBQQ0ECCmECjkIAAAAAAB8wCQYFKw4DAhoFAKBd
// SIG // MBgGCSqGSIb3DQEJAzELBgkqhkiG9w0BBwEwHAYJKoZI
// SIG // hvcNAQkFMQ8XDTEzMDMxNTA2MzQwMlowIwYJKoZIhvcN
// SIG // AQkEMRYEFOQht2KUFMgANQWyZRv9btvWYHwKMA0GCSqG
// SIG // SIb3DQEBBQUABIIBAF32QXjUULZ/gmgDqmOzl2COvM+T
// SIG // 3Fc3oedU8EgW+iKOjn3Giys7ARhhlw2LBHQeUGQY9lLw
// SIG // jmXP1ODpgqe4cfVoUdbYsKxm9TYGQej9C94B7mA9x9Q9
// SIG // 6VxT9wYFZPGO2G9jPCpta/+C6Y8rV4rMadVovkr2MGFy
// SIG // 2IoDk+/6cY5bcofOb2McpO9DDN7G9DEYkjCUE+vGa7/3
// SIG // tKQ+yKiw8b+VMDdXr3HazbZH+QpuL8JNxOeUBcEQUZpF
// SIG // 0+D39RaWTaVdanCVtVKxFAWxm3PNQ+8e+jTNXb9ijBrJ
// SIG // 2akvn7WHrrRBSNz/ML2V7QolGB1SgTSN691yty5z/Ol6
// SIG // fY+/ZFE=
// SIG // End signature block
